import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { MapPin, Calendar, Star, User, ScrollText, Clock, Award, BookOpen } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyTours.css'

const MyTours = () => {
  const [tours, setTours] = useState([])

  useEffect(() => {
    const saved = localStorage.getItem('excursionsTours')
    if (saved) {
      try {
        setTours(JSON.parse(saved))
      } catch (e) {
        setTours([])
      }
    } else {
      const defaultTours = [
        {
          id: 1,
          route: 'Тайны Красной площади',
          guide: 'Анна Петрова',
          date: '2025-04-15',
          rating: 5,
          duration: '3 часа',
          theme: 'Историческая',
          highlights: ['Кремль', 'Собор Василия Блаженного', 'ГУМ']
        },
        {
          id: 2,
          route: 'Гастрономические секреты Замоскворечья',
          guide: 'Михаил Соколов',
          date: '2025-04-12',
          rating: 4,
          duration: '2.5 часа',
          theme: 'Гастрономическая',
          highlights: ['Традиционные блюда', 'Исторические кафе', 'Дегустации']
        },
        {
          id: 3,
          route: 'Природные сокровища Коломенского',
          guide: 'Елена Волкова',
          date: '2025-04-10',
          rating: 5,
          duration: '4 часа',
          theme: 'Природная',
          highlights: ['Древние дубы', 'Яблоневые сады', 'Речные пейзажи']
        }
      ]
      setTours(defaultTours)
      localStorage.setItem('excursionsTours', JSON.stringify(defaultTours))
    }
  }, [])

  const totalTours = tours.length
  const averageRating = tours.length > 0
    ? (tours.reduce((sum, t) => sum + t.rating, 0) / tours.length).toFixed(1)
    : 0
  const upcomingTours = tours.filter(t => new Date(t.date) >= new Date()).length

  return (
    <div className="my-tours">
      <div className="container">
        <motion.section 
          className="hero-elegant"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <div className="hero-content-elegant">
            <h1 className="hero-title-elegant">Мои экскурсии</h1>
            <div className="hero-divider"></div>
            <p className="hero-description-elegant">
              Погрузитесь в богатую историю и культуру. Отслеживайте ваши 
              запланированные и завершенные экскурсии, открывая для себя 
              наследие прошлого.
            </p>
          </div>
        </motion.section>

        <motion.section 
          className="stats-elegant"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <div className="stats-elegant-list">
            <div className="stat-elegant-item">
              <div className="stat-elegant-icon">
                <BookOpen size={22} strokeWidth={1.5} />
              </div>
              <div className="stat-elegant-content">
                <div className="stat-elegant-value">{totalTours}</div>
                <div className="stat-elegant-label">Экскурсий</div>
              </div>
            </div>
            <div className="stat-elegant-item">
              <div className="stat-elegant-icon">
                <Star size={22} strokeWidth={1.5} />
              </div>
              <div className="stat-elegant-content">
                <div className="stat-elegant-value">{averageRating}</div>
                <div className="stat-elegant-label">Рейтинг</div>
              </div>
            </div>
            <div className="stat-elegant-item">
              <div className="stat-elegant-icon">
                <Calendar size={22} strokeWidth={1.5} />
              </div>
              <div className="stat-elegant-content">
                <div className="stat-elegant-value">{upcomingTours}</div>
                <div className="stat-elegant-label">Предстоящих</div>
              </div>
            </div>
            <div className="stat-elegant-item">
              <div className="stat-elegant-icon">
                <Award size={22} strokeWidth={1.5} />
              </div>
              <div className="stat-elegant-content">
                <div className="stat-elegant-value">{tours.filter(t => t.rating === 5).length}</div>
                <div className="stat-elegant-label">Отлично</div>
              </div>
            </div>
          </div>
        </motion.section>

        <ImageSlider />

        <motion.section 
          className="tours-elegant"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.3 }}
        >
          <div className="section-header-elegant">
            <h2 className="section-title-elegant">Мои экскурсии</h2>
            <div className="section-divider"></div>
          </div>
          {tours.length === 0 ? (
            <div className="empty-elegant">
              <div className="empty-icon-elegant">
                <ScrollText size={48} strokeWidth={1.5} />
              </div>
              <h3 className="empty-title-elegant">Начните свое путешествие</h3>
              <p className="empty-text-elegant">Запишитесь на первую экскурсию и откройте для себя культурное наследие</p>
            </div>
          ) : (
            <div className="tours-timeline">
              {tours.map((tour, index) => (
                <motion.div
                  key={tour.id}
                  className="tour-timeline-item"
                  initial={{ opacity: 0, x: -30 }}
                  animate={{ opacity: 1, x: 0 }}
                  transition={{ delay: index * 0.15 }}
                >
                  <div className="timeline-marker">
                    <div className="marker-dot"></div>
                    <div className="marker-line"></div>
                  </div>
                  <div className="tour-timeline-content">
                    <div className="tour-timeline-header">
                      <div className="tour-theme-elegant">{tour.theme}</div>
                      <div className="tour-rating-elegant">
                        {Array.from({ length: 5 }).map((_, i) => (
                          <Star
                            key={i}
                            size={14}
                            strokeWidth={1.5}
                            fill={i < tour.rating ? '#D4A574' : 'none'}
                            color={i < tour.rating ? '#D4A574' : '#D4D4D4'}
                          />
                        ))}
                      </div>
                    </div>
                    <h3 className="tour-route-elegant">{tour.route}</h3>
                    <div className="tour-details-elegant">
                      <div className="tour-detail-elegant">
                        <User size={14} strokeWidth={1.5} />
                        <span>{tour.guide}</span>
                      </div>
                      <div className="tour-detail-elegant">
                        <Calendar size={14} strokeWidth={1.5} />
                        <span>{new Date(tour.date).toLocaleDateString('ru-RU', { 
                          day: 'numeric', 
                          month: 'long', 
                          year: 'numeric' 
                        })}</span>
                      </div>
                      <div className="tour-detail-elegant">
                        <Clock size={14} strokeWidth={1.5} />
                        <span>{tour.duration}</span>
                      </div>
                    </div>
                    {tour.highlights && (
                      <div className="tour-highlights-elegant">
                        {tour.highlights.map((highlight, i) => (
                          <span key={i} className="highlight-elegant">{highlight}</span>
                        ))}
                      </div>
                    )}
                  </div>
                </motion.div>
              ))}
            </div>
          )}
        </motion.section>

        <motion.section 
          className="heritage-section"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <div className="section-header-elegant">
            <h2 className="section-title-elegant">Культурное наследие</h2>
            <div className="section-divider"></div>
          </div>
          <div className="heritage-grid">
            <div className="heritage-item">
              <div className="heritage-icon">
                <ScrollText size={28} strokeWidth={1.5} />
              </div>
              <h3 className="heritage-title">Исторические памятники</h3>
              <p className="heritage-text">
                Исследуйте архитектурные шедевры, которые рассказывают 
                историю целых эпох и поколений.
              </p>
            </div>
            <div className="heritage-item">
              <div className="heritage-icon">
                <BookOpen size={28} strokeWidth={1.5} />
              </div>
              <h3 className="heritage-title">Культурные традиции</h3>
              <p className="heritage-text">
                Погрузитесь в мир традиций, обычаев и обрядов, 
                передаваемых из поколения в поколение.
              </p>
            </div>
            <div className="heritage-item">
              <div className="heritage-icon">
                <MapPin size={28} strokeWidth={1.5} />
              </div>
              <h3 className="heritage-title">Уникальные места</h3>
              <p className="heritage-text">
                Откройте для себя скрытые уголки города, 
                недоступные в обычных туристических маршрутах.
              </p>
            </div>
            <div className="heritage-item">
              <div className="heritage-icon">
                <Award size={28} strokeWidth={1.5} />
              </div>
              <h3 className="heritage-title">Экспертное знание</h3>
              <p className="heritage-text">
                Наши гиды обладают глубокими знаниями истории, 
                культуры и искусства.
              </p>
            </div>
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default MyTours

